        private bool VerifyJwtSignature(string jwt, string publicKeyPem)
        {
            string[] parts = jwt.Split('.');
            if (parts.Length != 3)
            {
                Console.WriteLine("Invalid JWT format.");
                return false;
            }

            string headerPayload = $"{parts[0]}.{parts[1]}"; // Data to verify
            string signature = parts[2];

            byte[] signatureBytes = Base64UrlDecode(signature);
            byte[] dataBytes = Encoding.UTF8.GetBytes(headerPayload);

            try
            {
                // Load RSA public key using BouncyCastle
                RSA rsa = LoadRsaPublicKeyFromPem(publicKeyPem);

                // Verify the signature using SHA256 and RSA
                using (RSACryptoServiceProvider rsaProvider = new RSACryptoServiceProvider())
                {
                    rsaProvider.ImportParameters(rsa.ExportParameters(false));
                    return rsaProvider.VerifyData(dataBytes, "SHA256", signatureBytes);
                }
            }
            catch (Exception ex)
            {
                Console.WriteLine($"Error verifying JWT: {ex.Message}");
                return false;
            }
        }

        private RSA LoadRsaPublicKeyFromPem(string publicKeyPem)
        {
            using (var reader = new System.IO.StringReader(publicKeyPem))
            {
                var pemReader = new PemReader(reader);
                var keyParameter = (RsaKeyParameters)pemReader.ReadObject();
                RSA rsa = RSA.Create();
                rsa.ImportParameters(DotNetUtilities.ToRSAParameters(keyParameter));
                return rsa;
            }
        }

        private byte[] Base64UrlDecode(string input)
        {
            string output = input.Replace('-', '+').Replace('_', '/');
            while (output.Length % 4 != 0)
            {
                output += "=";
            }
            return Convert.FromBase64String(output);
        }

        private async Task<RSA?> GetRsaPublicKeyFromJwks(string url)
        {
            using HttpClient client = new HttpClient();
            string jwksJson = await client.GetStringAsync(url);

            using JsonDocument doc = JsonDocument.Parse(jwksJson);
            JsonElement keys = doc.RootElement.GetProperty("keys");

            foreach (JsonElement key in keys.EnumerateArray())
            {
                if (key.GetProperty("kty").GetString() == "RSA" &&
                    key.GetProperty("use").GetString() == "sig")
                {
                    string n = key.GetProperty("n").GetString(); // Modulus
                    string e = key.GetProperty("e").GetString(); // Exponent

                    byte[] modulus = Base64UrlDecode(n);
                    byte[] exponent = Base64UrlDecode(e);

                    RSAParameters rsaParams = new RSAParameters
                    {
                        Modulus = modulus,
                        Exponent = exponent
                    };

                    RSA rsa = RSA.Create();
                    rsa.ImportParameters(rsaParams);
                    return rsa;
                }
            }

            return null;
        }

        private string ConvertRsaToPem(RSA rsa)
        {
            // Convert RSA to BouncyCastle key format
            AsymmetricKeyParameter keyParam = DotNetUtilities.GetRsaPublicKey(rsa);

            // Use StringWriter to capture the PEM output
            using (var stringWriter = new StringWriter())
            {
                var pemWriter = new PemWriter(stringWriter);
                pemWriter.WriteObject(keyParam);

                return stringWriter.ToString(); // Return PEM-formatted string
            }
        }